#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/ioctl.h>

#include "util-pie.h"
#include "proc_parse.h"
#include "cr_options.h"
#include "fdset.h"
#include "protobuf.h"
#include "files.h"
#include "files-reg.h"
#include "testfd.h"

#include "protobuf/testfd.pb-c.h"

#define TEST_DEV_PATH	"/dev/test"

struct testfd_dump_arg {
	u32			id;
	const struct fd_parms	*p;
};

struct testfd_info {
	TestfdEntry		*testfe;
	struct file_desc	d;
};

int check_testfd(void)
{
	int fd, ret = 0;

	if (opts.check_ms_kernel) {
		pr_warn("Skipping test support check\n");
		return 0;
	}

	fd = open(TEST_DEV_PATH, O_RDWR);
	if (fd < 0) {
		pr_perror("Can't check test support");
		return 0;
	}

	close(fd);
	return ret;
}

static int dump_testfd_entry(union fdinfo_entries *e, void *arg)
{
	struct testfd_dump_arg *da = arg;
	TestfdEntry *testfe = &e->testfe;

	testfe->id		= da->id;
	testfe->flags	= da->p->flags;

	pr_info("Dumping TEST FD id %#x\n", testfe->id);

	return pb_write_one(fdset_fd(glob_fdset, CR_FD_TESTFD), &e->testfe, PB_TESTFD);
}

static int dump_one_testfd(int lfd, u32 id, const struct fd_parms *p)
{
	struct testfd_dump_arg da = { .id = id, .p = p, };
	return parse_fdinfo(lfd, FD_TYPES__TESTFD, dump_testfd_entry, &da);
}

const struct fdtype_ops testfd_dump_ops = {
	.type		= FD_TYPES__TESTFD,
	.dump		= dump_one_testfd,
};

static int testfd_open(struct file_desc *d)
{
	struct testfd_info *info = container_of(d, struct testfd_info, d);
	TestfdEntry *testfe = info->testfe;
	FownEntry fe = FOWN_ENTRY__INIT;

	pr_info("Creating TEST FD id %#x\n", testfe->id);
	int tmp = open(TEST_DEV_PATH, O_RDWR);
	if (tmp < 0) {
		pr_perror("Can't create TEST FD for %#x", testfe->id);
		return -1;
	}
	pr_info("restoring TEST FD\n");

	if (rst_file_params(tmp, &fe, testfe->flags)) {
		pr_perror("Can't restore params for TEST FD %#x", testfe->id);
		goto err_close;
	}
	return tmp;

err_close:
	close_safe(&tmp);
	return -1;
}

static struct file_desc_ops testfd_desc_ops = {
	.type		= FD_TYPES__TESTFD,
	.open		= testfd_open,
};

static int collect_one_testfd(void *o, ProtobufCMessage *msg)
{
	struct testfd_info *info = o;

	info->testfe = pb_msg(msg, TestfdEntry);
	return file_desc_add(&info->d, info->testfe->id, &testfd_desc_ops);
}

struct collect_image_info testfd_cinfo = {
	.fd_type	= CR_FD_TESTFD,
	.pb_type	= PB_TESTFD,
	.priv_size	= sizeof(struct testfd_info),
	.collect	= collect_one_testfd,
	.flags		= COLLECT_OPTIONAL,
};


